package com.agilex.healthcare.veteranappointment.resources;

import java.util.Date;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.veteranappointment.dataservice.*;
import com.agilex.healthcare.veteranappointment.domain.*;
import gov.va.vamf.scheduling.direct.datalayer.appointment.PatientIdentifiersDataService;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import gov.va.vamf.mdws.client.v1.mobilehealthplatform.domain.MhpUser;
import gov.va.vamf.security.v1.VamfUser;
import gov.va.vamf.security.v1.filters.JwtRbacRestricted;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import gov.va.vamf.security.v1.filters.JwtResourceRestricted;

import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.veteranappointment.datalayer.feedback.UserFeedbackDataService;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.veteranappointment.uri.VARPatientResourceDirectoryBuilder;
import com.agilex.healthcare.veteranappointment.utils.PilotFacilitiesHelper;
import com.agilex.healthcare.veteranappointment.utils.linkbuilder.AppointmentRequestLinkBuilder;
import com.agilex.healthcare.veteranappointment.utils.linkbuilder.AppointmentRequestMessagesLinkBuilder;

@Path("/appointment-service")
@Component
@Scope("request")
public class AppointmentRequestResource extends AbstractUserResource {

    @Resource
    VARAppointmentsMetadata appointmentsMetadata;

    @Resource
    PilotFacilitiesHelper pilotFacilitiesHelper;

    @Resource
    String haMode;

    @Resource
    DetailCodeDataService detailCodeDataService;

    @Resource
    AppointmentRequestDataService appointmentRequestDataService;

    @Resource
    CCAppointmentRequestDataService ccAppointmentRequestDataService;

    @Resource
    AppointmentRequestInProcessDataService appointmentRequestInProcessDataService;

    @Resource
    PatientMetadataDataService patientMetadataDataService;

    @Resource
    VARPatientResourceDirectoryBuilder varPatientResourceDirectoryBuilder;

    @Resource
    UserFeedbackDataService userFeedbackDataService;

    @Resource
    PatientIdentifiersDataService patientIdentifiersDataService;

    /**
     * This REST service provides resource directory to front-end
     * @param uriInfo
     * @param patientId
     * @param assigningAuthority
     * @return ResourceDirectory
     */
    @GET
    @Path("/var-patient-resource-directory")
    @Produces({"application/xml","application/json"})
    @JwtRbacRestricted("Veteran")
    public ResourceDirectory getVarPatientResourceDirectory(@Context UriInfo uriInfo,
                                                            @QueryParam("patientId") String patientId,
                                                            @QueryParam("assigningAuthority") String assigningAuthority){
    	ResourceDirectory resourceDirectory = varPatientResourceDirectoryBuilder.getResourceDirectory(uriInfo.getBaseUri(), patientId, assigningAuthority);
    	return resourceDirectory;
    }

    /**
     * This REST service is used to fetch detail codes from SED database
     * @return VARAppointmentsMetadata
     */
    @GET
    @Path("/metadata")
    @Produces({"application/xml", "application/json"})
    @JwtRbacRestricted("Veteran")
    public VARAppointmentsMetadata getAppointmentsMetadata() {
        Set<VARDetailCode> detailCodes = detailCodeDataService.fetchDetailCodes();
        appointmentsMetadata.setDetailCodes(detailCodes);
        return appointmentsMetadata;
    }

    /**
     * A resource that returns all {@link VARAppointmentRequests} that have been
     * created by this user.
     *
     * @return Returns {@link VARAppointmentRequests} associated with the user
     *         calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointments")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequests getAppointmentRequests(@Context ContainerRequestContext context,
                                                         @Context UriInfo uriInfo,
                                                         @PathParam("assigning-authority") String assigningAuthority,
                                                         @PathParam("patient-id") String patientId) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        
        DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);
        VARAppointmentRequests appointmentRequests;

        appointmentRequests = appointmentRequestDataService.getPatientAppointmentRequests(patientIdentifier, dateFilter, ScopeFilter.getInstanceForLongitudinalScope());

        appointmentRequests.setLastAccessDate(patientMetadataDataService.fetchAndUpdateLastAccessDate(patientIdentifier, ScopeFilter.getInstanceForLongitudinalScope()));

        VamfUser vamfUser = getCurrentVamfUser(context);

        updateAtomLinks(uriInfo, appointmentRequests);

        return appointmentRequests;
    }

    /**
     * A resource that returns all {@link VARAppointmentRequestMessages} that have
     * been created by this user.
     *
     * @return Returns {@link VARAppointmentRequestMessages} associated with the
     *         user calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequestMessages getAppointmentRequestMessages(@PathParam("assigning-authority") String assigningAuthority, @PathParam("patient-id") String patientId, @PathParam("system-id") String systemId,
                                                                    @PathParam("appointment-request-id") String appointmentRequestId, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);

        VARAppointmentRequestMessages messages = appointmentRequestDataService.fetchAppointmentRequestMessages(patientIdentifier, dataIdentifier);

        updateAtomLinks(uriInfo, messages);

        return messages;
    }

    /**
     * This REST service is used to save a message for select appointment request
     * @param assigningAuthority
     * @param patientId
     * @param appointmentRequestId
     * @param systemId
     * @param appointmentRequestMessage
     * @return VARAppointmentRequestMessage
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequestMessage createAppointmentRequestMessage(@Context ContainerRequestContext context,
                                                                        @PathParam("assigning-authority") String assigningAuthority,
                                                                        @PathParam("patient-id") String patientId,
                                                                        @PathParam("appointment-request-id") String appointmentRequestId,
                                                                        @PathParam("system-id") String systemId,
                                                                        VARAppointmentRequestMessage appointmentRequestMessage) {

        if (appointmentRequestId.equals(appointmentRequestMessage.getAppointmentRequestId())) {
            appointmentRequestMessage.setPatientIdentifier(new PatientIdentifier(assigningAuthority, patientId));
            appointmentRequestMessage.setDataIdentifier(new DataIdentifier(systemId, appointmentRequestId));

            VARAppointmentRequestMessage result = appointmentRequestDataService.saveAppointmentRequestMessage(appointmentRequestMessage, getCurrentVamfUser(context));

            return result;
        } else {
            return null;
        }
    }

    /**
     * This REST service is used to save/update the flag that message from provider is been read by veteran
     * @param assigningAuthority
     * @param patientId
     * @param systemId
     * @param appointmentRequestId
     * @return Response
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointment-requests/system/{system-id}/id/{appointment-request-id}/messages/read")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public Response markMessagesAsRead(@Context ContainerRequestContext context,
                                       @PathParam("assigning-authority") String assigningAuthority,
                                       @PathParam("patient-id") String patientId,
                                       @PathParam("system-id") String systemId,
                                       @PathParam("appointment-request-id") String appointmentRequestId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);
        VamfUser vamfUser = getCurrentVamfUser(context);

        VARAppointmentRequest appointmentRequest = appointmentRequestDataService.updateAppointmentMessageFlag(appointmentRequestId,patientIdentifier,dataIdentifier, vamfUser);
        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setDataIdentifier(dataIdentifier);
        if (appointmentRequest.getAppointmentRequestId().equals(appointmentRequestId)) {
            return Response.ok().build();
        } else {
            return Response.serverError().build();
        }
    }

    private void updateAtomLinks(UriInfo uriInfo, VARAppointmentRequestMessages messages) {
        if (uriInfo != null) {
            AppointmentRequestMessagesLinkBuilder linkBuilder = new AppointmentRequestMessagesLinkBuilder(uriInfo.getBaseUri());
            linkBuilder.fillLinks(messages, uriInfo.getRequestUri());
        }
    }

    private void updateAtomLinks(UriInfo uriInfo, VARAppointmentRequests appointmentRequests) {
        if (uriInfo != null) {
            AppointmentRequestLinkBuilder linkBuilder = new AppointmentRequestLinkBuilder(uriInfo.getBaseUri());
            linkBuilder.fillLinks(appointmentRequests, uriInfo.getRequestUri());
        }
    }

    /**
     * A resource that returns a single {@link VARAppointmentRequest} by user and
     * id
     *
     * @return Returns {@link VARAppointmentRequest} associated with the id and
     *         user calling the request
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequest getAppointmentRequest(@PathParam("assigning-authority") String assigningAuthority, @PathParam("system-id") String systemId, @PathParam("patient-id") String patientId,
                                                    @PathParam("appointment-request-id") String appointmentRequestId, @Context UriInfo uriInfo) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);
        VARAppointmentRequest appointmentRequest = null;

        appointmentRequest = appointmentRequestDataService.getPatientAppointmentRequest(patientIdentifier, dataIdentifier);
        VARAppointmentRequestInProcess beingProcessedBy = retrieveAppointmentRequestInProcess(appointmentRequest.getUniqueId());
        appointmentRequest.setBeingProcessedBy(beingProcessedBy);

        updateAtomLinks(appointmentRequest, uriInfo);

        return appointmentRequest;
    }

    private VARAppointmentRequestInProcess retrieveAppointmentRequestInProcess(String appointmentRequestId) {
        return appointmentRequestInProcessDataService.fetchAppointmentRequestInProcessById(appointmentRequestId);
    }

    /**
     * A resource that saves a single {@link VARAppointmentRequest} by user and id
     *
     * @return Returns {@link VARAppointmentRequest} associated with the id and
     *         user calling the request
     */
    @PUT
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequest updateAppointmentRequest(@Context ContainerRequestContext context,
                                                          @Context HttpServletRequest request,
                                                          @Context UriInfo uriInfo,
                                                          @PathParam("assigning-authority") String assigningAuthority,
                                                          @PathParam("system-id") String systemId,
                                                          @PathParam("patient-id") String patientId,
                                                          @PathParam("appointment-request-id") String appointmentRequestId,
                                                          VARAppointmentRequest appointmentRequest) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        Date facilityDt = appointmentRequestDataService.getFacilityDt(appointmentRequest);
        DataIdentifier dataIdentifier = new DataIdentifier(systemId, appointmentRequestId);

        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setDataIdentifier(dataIdentifier);
        appointmentRequest.setLastUpdatedDate(facilityDt);

        for (VARAppointmentRequestDetailCode ardc : appointmentRequest.getAppointmentRequestDetailCode()) {
            if (ardc.getCreatedDate() == null) {
                ardc.setCreatedDate(facilityDt);
            }
        }

        boolean fullValidation = false;
        appointmentRequest = saveAppointmentRequest(appointmentRequest, fullValidation, uriInfo, getCurrentVamfUser(context));

        return appointmentRequest;
    }

    private void setCreateDateInDetailCodesOfParentAppointmentRequest(VARAppointmentRequest appointmentRequest) {
        VARAppointmentRequest parentAppointmentRequest = appointmentRequest.getParentRequest();
        Date facilityDt = appointmentRequestDataService.getFacilityDt(appointmentRequest);

        if (parentAppointmentRequest != null) {
            for (VARAppointmentRequestDetailCode ardc : parentAppointmentRequest.getAppointmentRequestDetailCode()) {
                if (ardc.getCreatedDate() == null) {
                    ardc.setCreatedDate(facilityDt);
                }
            }
        }
    }

    /**
     * A resource that allows a client to save an {@link VARAppointmentRequest}.
     * See {@link AppointmentRequestValidator} for val idation logic.
     *
     * @param VARAppointmentRequest
     * @return The {@link VARAppointmentRequest} that was saved.
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointments")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public VARAppointmentRequest submitAppointmentRequest(@Context ContainerRequestContext context,
                                                          @Context HttpServletRequest request,
                                                          @Context UriInfo uriInfo,
                                                          @PathParam("assigning-authority") String assigningAuthority,
                                                          @PathParam("patient-id") String patientId,
                                                          VARAppointmentRequest appointmentRequest) {
        VamfUser vamfUser = getCurrentVamfUser(context);
        prepareAppointmentRequestForSubmission(appointmentRequest, assigningAuthority, patientId);

        boolean fullValidation = true;
        setCreateDateInDetailCodesOfParentAppointmentRequest(appointmentRequest);
        appointmentRequest = saveAppointmentRequest(appointmentRequest, fullValidation, uriInfo, vamfUser);

        return appointmentRequest;
    }


    /**
     * A resource that allows a client to save an {@link CCAppointmentRequest}.
     * See {@link AppointmentRequestValidator} for validation logic.
     *
     * @param CCAppointmentRequest
     * @return The {@link CCAppointmentRequest} that was saved.
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/community-care-appointment")
    @Consumes({"application/xml", "application/json"})
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public CCAppointmentRequest submitAppointmentRequest(@Context ContainerRequestContext context,
                                                         @Context HttpServletRequest request,
                                                         @Context UriInfo uriInfo,
                                                         @PathParam("assigning-authority") String assigningAuthority,
                                                         @PathParam("patient-id") String patientId,
                                                         CCAppointmentRequest ccAppointmentRequest) {
        VamfUser vamfUser = getCurrentVamfUser(context);

        prepareAppointmentRequestForSubmission(ccAppointmentRequest, assigningAuthority, patientId);

        boolean fullValidation = true;
        setCreateDateInDetailCodesOfParentAppointmentRequest(ccAppointmentRequest);
        ccAppointmentRequest = saveCCAppointmentRequest(ccAppointmentRequest, fullValidation, uriInfo, vamfUser);

        return ccAppointmentRequest;
    }

    /**
     * This REST service is used to save the state of appointment request thats under process of being booked by a provider
     * so that other provider cannot update the appointment request
     * @param VARAppointmentRequestInProcess
     * @return The {@link VARAppointmentRequestInProcess} that was saved.
     */
    @POST
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}/inprocess")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public Response saveAppointmentRequestInProcess(@Context ContainerRequestContext context,
                                                    @PathParam("patient-id") String patientId,
                                                    @PathParam("system-id") String systemId,
                                                    @PathParam("appointment-request-id") String appointmentRequestId) {
        Response response;
        VARAppointmentRequestInProcess beingProcessedBy;
        beingProcessedBy = retrieveAppointmentRequestInProcess(appointmentRequestId);

        if (beingProcessedBy == null) {
            MhpUser mhpUser = getCurrentUser(context);
            beingProcessedBy = new VARAppointmentRequestInProcess();
            beingProcessedBy.setAppointmentRequestId(appointmentRequestId);
            beingProcessedBy.setUserId(mhpUser.getId());
            beingProcessedBy.setFirstName(mhpUser.getFirstName());
            beingProcessedBy.setLastName(mhpUser.getLastName());

            beingProcessedBy = appointmentRequestInProcessDataService.saveAppointmentRequestInProcess(beingProcessedBy);

            response = Response.ok(beingProcessedBy).build();
        } else if (beingProcessedBy.getUserId().equals(getCurrentUserId(context))) {
            response = Response.ok(beingProcessedBy).build();
        } else {
            response = Response.status(400).entity(beingProcessedBy).build();
        }

        return response;
    }

    /**
     * This REST service is to used to delete the state of the appointment requests thats processed/booked by a provider
     * @param systemId, appointmentRequestId
     * @return The {@link VARAppointmentRequestInProcess} that was saved.
     */
    @DELETE
    @Path("/patient/{assigning-authority}/{patient-id}/appointments/system/{system-id}/id/{appointment-request-id}/inprocess")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public Response deleteAppointmentRequestInProcess(@PathParam("system-id") String systemId, @PathParam("appointment-request-id") String appointmentRequestId) {

    	appointmentRequestInProcessDataService.deleteAppointmentRequestInProcessById(appointmentRequestId);

        return Response.ok().build();
    }

    private VARAppointmentRequest saveAppointmentRequest(VARAppointmentRequest appointmentRequest, boolean fullValidation, UriInfo uriInfo, VamfUser vamfUser) {
            appointmentRequest = appointmentRequestDataService.saveAppointmentRequest(appointmentRequest, haMode, fullValidation, vamfUser);
            updateAtomLinks(appointmentRequest, uriInfo);
            return appointmentRequest;
    }

    private CCAppointmentRequest saveCCAppointmentRequest(CCAppointmentRequest ccAppointmentRequest, boolean fullValidation,
                                                          UriInfo uriInfo, VamfUser vamfUser) {
        ccAppointmentRequest = ccAppointmentRequestDataService.saveCCAppointmentRequest(ccAppointmentRequest, haMode, fullValidation, vamfUser);
        updateAtomLinks(ccAppointmentRequest, uriInfo);
        return ccAppointmentRequest;
    }

    private void updateAtomLinks(VARAppointmentRequest appointmentRequest, UriInfo uriInfo) {
        AppointmentRequestLinkBuilder linkBuilder = new AppointmentRequestLinkBuilder(uriInfo.getBaseUri());
        linkBuilder.fillLinks(appointmentRequest, uriInfo.getRequestUri());
    }

    private void prepareAppointmentRequestForSubmission(VARAppointmentRequest appointmentRequest, String assigningAuthority, String patientId) {
        Date facilityDt = appointmentRequestDataService.getFacilityDt(appointmentRequest);

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        appointmentRequest.setPatientIdentifier(patientIdentifier);
        appointmentRequest.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
        appointmentRequest.setCreatedDate(facilityDt);
        appointmentRequest.setLastUpdatedDate(facilityDt);
        appointmentRequest.setDeletedDate(null);
        if (!appointmentRequest.isTextMessagingAllowed()) {
            appointmentRequest.setTextMessagingPhoneNumber("");
        }
    }

    /**
     * This REST service is used to save feedback from the user
     * @param userFeedback
     * @return UserFeedback
     */
    @POST
	@Path("/feedback")
	@Consumes({ "application/xml", "application/json" })
    @JwtRbacRestricted("Veteran")
	public VARFeedback recordFeedbackOnAppointmentRequest(VARFeedback userFeedback) {
		return userFeedbackDataService.saveUserFeedback(userFeedback);
	}
}
